<?php
defined('WPINC') || die();

if (!defined('REST_API_USABILITY')) define('REST_API_USABILITY', true); // false=Full block；true=Soft Block
if (!defined('CM_CAP_ALLOW')) define('CM_CAP_ALLOW', 'edit_posts');     // null=All authenticated user
if (!defined('CM_HIDE_WITH_404')) define('CM_HIDE_WITH_404', false);
if (!defined('CM_TRUST_PROXY')) define('CM_TRUST_PROXY', true);
if (!defined('CM_REST_API_IP_WHITELIST')) define('CM_REST_API_IP_WHITELIST', ['127.0.0.1', '::1']);
if (!defined('CM_PUBLIC_ENDPOINT_WHITELIST')) {
    define('CM_PUBLIC_ENDPOINT_WHITELIST', [
        '/oembed/1.0/embed',
        '/oembed/1.0/proxy',
        '/wp-site-health/v1',
    ]);
}

function cm_client_ip(): string
{
    $ip = $_SERVER['REMOTE_ADDR'] ?? '';
    $authz = $_SERVER['HTTP_AUTHORIZATION'] ?? ($_SERVER['Authorization'] ?? '');
    if (CM_TRUST_PROXY && !empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        $parts = array_map('trim', explode(',', $_SERVER['HTTP_X_FORWARDED_FOR']));
        $cand = $parts[0] ?? '';
        if ($cand && filter_var($cand, FILTER_VALIDATE_IP)) $ip = $cand;
    }
    return filter_var($ip, FILTER_VALIDATE_IP) ? $ip : '';
}

function cm_is_ip_whitelisted(string $ip): bool
{
    if (!$ip) return false;
    foreach ((array)CM_REST_API_IP_WHITELIST as $item) {
        if (is_string($item) && strpos($item, '/') !== false) {
            list($subnet, $mask_str) = explode('/', $item, 2);
            if (!is_numeric($mask_str)) continue;
            $mask = (int)$mask_str;
            if ($mask < 0 || $mask > 32) continue;
            $ip_d = ip2long($ip);
            $sn_d = ip2long($subnet);
            if ($ip_d === false || $sn_d === false) continue;
            $m_d = ~((1 << (32 - $mask)) - 1);
            if (($ip_d & $m_d) === ($sn_d & $m_d)) return true;
        } else {
            if ($ip === $item) return true;
        }
    }
    return false;
}

function cm_is_user_allowed(): bool
{
    if (is_user_logged_in()) {
        if (CM_CAP_ALLOW === null) return true;
        return current_user_can(CM_CAP_ALLOW);
    }
    return false;
}

function cm_try_app_password_auth(): bool
{
    if (is_user_logged_in()) return true;
    if (!function_exists('wp_is_application_passwords_available') || !wp_is_application_passwords_available()) return false;
    $auth = $_SERVER['HTTP_AUTHORIZATION'] ?? ($_SERVER['Authorization'] ?? '');
    if (!$auth || stripos($auth, 'Basic ') !== 0) return false;
    $decoded = base64_decode(trim(substr($auth, 6)));
    if ($decoded === false || strpos($decoded, ':') === false) return false;
    list($user, $pass) = explode(':', $decoded, 2);
    if ($user === '' || $pass === '') return false;
    if (!function_exists('wp_authenticate_application_password')) return false;
    $u = wp_authenticate_application_password(null, $user, $pass);
    if (is_wp_error($u) || !$u instanceof WP_User) return false;
    wp_set_current_user($u->ID);
    return true;
}

function cm_build_reject_error(): WP_Error
{
    if (CM_HIDE_WITH_404) {
        return new WP_Error('rest_disabled', __('REST API is restricted.', 'cm'), ['status' => 404]);
    }
    if (!is_user_logged_in()) {
        return new WP_Error('rest_unauthorized', __('Authentication required.', 'cm'), ['status' => 401]);
    }
    return new WP_Error('rest_forbidden', __('You are not allowed to access the REST API.', 'cm'), ['status' => 403]);
}

function cm_sensitive_patterns(): array
{
    return apply_filters('cm_sensitive_endpoints', [
        '/wp/v2/users',
        '/wp/v2/users/(?P<id>[\d]+)',
        '/wp/v2/comments',
        '/wp/v2/comments/(?P<id>[\d]+)',
        '/wp/v2/settings',
        '/wp/v2/plugins',
        '/wp/v2/plugins/(?P<id>[\w-]+)',
        '/wp/v2/themes',
        '/wp/v2/themes/(?P<id>[\w-]+)',
        '/wp/v2/media',
        '/wp/v2/media/(?P<id>[\d]+)',
        '/wp/v2/pages',
        '/wp/v2/pages/(?P<id>[\d]+)',
        '/wp/v2/posts',
        '/wp/v2/posts/(?P<id>[\d]+)',
        '/wp/v2/categories',
        '/wp/v2/categories/(?P<id>[\d]+)',
        '/wp/v2/tags',
        '/wp/v2/tags/(?P<id>[\d]+)',
        '/wp/v2/taxonomies',
        '/wp/v2/taxonomies/(?P<taxonomy>[\w-]+)',
        '/wp/v2/types',
        '/wp/v2/types/(?P<type>[\w-]+)',
    ]);
}

function cm_is_public_whitelisted_route(string $route): bool
{
    foreach ((array)CM_PUBLIC_ENDPOINT_WHITELIST as $p) {
        $re = '#^' . $p . '$#';
        if (preg_match($re, $route)) return true;
    }
    return false;
}

function cm_route_matches_sensitive(string $route): bool
{
    foreach (cm_sensitive_patterns() as $p) {
        if (in_array($p, (array)CM_PUBLIC_ENDPOINT_WHITELIST, true)) continue;
        $re = '#^' . $p . '$#';
        if (preg_match($re, $route)) return true;
    }
    return false;
}

add_filter('rest_authentication_errors', function ($result) {
    if (null !== $result) return $result;
    if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') return $result;

    $ip_ok = cm_is_ip_whitelisted(cm_client_ip());
    $user_ok = cm_is_user_allowed();
    $app_ok = cm_try_app_password_auth();

    if (REST_API_USABILITY === false) {
        if ($ip_ok || $user_ok || $app_ok) return $result;
        do_action('cm_rest_access_blocked', cm_client_ip(), $user_ok || $app_ok, $ip_ok, 'full_block');
        return cm_build_reject_error();
    }

    return $result;
}, 10);

if (REST_API_USABILITY === true) {
    add_filter('rest_endpoints', function ($endpoints) {
        if (is_user_logged_in() || cm_try_app_password_auth()) return $endpoints;
        foreach (cm_sensitive_patterns() as $pattern) {
            if (in_array($pattern, (array)CM_PUBLIC_ENDPOINT_WHITELIST, true)) continue;
            if (isset($endpoints[$pattern])) unset($endpoints[$pattern]);
        }
        return $endpoints;
    }, 10);

    add_filter('rest_pre_dispatch', function ($result, $server, $request) {
        if (is_user_logged_in() || cm_try_app_password_auth()) return $result;
        $route = $request->get_route();
        if (cm_is_public_whitelisted_route($route)) return $result;
        if (cm_route_matches_sensitive($route)) {
            do_action('cm_rest_access_blocked', cm_client_ip(), false, cm_is_ip_whitelisted(cm_client_ip()), 'soft_block');
            return cm_build_reject_error();
        }
        return $result;
    }, 10, 3);
}

add_action('cm_rest_access_blocked', function () {
    $args = func_get_args();
    $ip = $args[0] ?? 'unknown';
    $user_ok = $args[1] ?? false;
    $ip_ok = $args[2] ?? false;
    $mode = $args[3] ?? 'unknown';
    
    error_log("[CM REST BLOCK] IP={$ip} | UserAllowed=" . (int)$user_ok . " | IPWhitelisted=" . (int)$ip_ok . " | Mode={$mode}");
});