<?php
if (!defined('ABSPATH')) {
    exit;
}

class CM_HIDE_LOGIN_URL
{
    private $wp_login_php = false;

    public function init()
    {
        add_action('init', [$this, 'on_init'], 9999);
        add_action('wp_loaded', [$this, 'on_wp_loaded']);
        add_action('setup_theme', [$this, 'on_setup_theme'], 1);

        add_filter('site_url', [$this, 'filter_site_url'], 10, 4);
        add_filter('network_site_url', [$this, 'filter_network_site_url'], 10, 3);
        add_filter('wp_redirect', [$this, 'filter_wp_redirect'], 10, 2);
        add_filter('site_option_welcome_email', [$this, 'filter_welcome_email']);
        add_filter('login_url', [$this, 'filter_login_url'], 10, 3);
        add_filter('user_request_action_email_content', [$this, 'filter_user_request_action_email_content'], 999, 2);

        remove_action('template_redirect', 'wp_redirect_admin_locations', 1000);
        add_action('template_redirect', [$this, 'redirect_export_data']);
    }

    private function use_trailing_slashes(): bool
    {
        $ps = get_option('permalink_structure');
        if (!$ps) {
            return false;
        }
        return (substr($ps, -1) === '/');
    }

    private function with_trailingslash(string $string): string
    {
        return $this->use_trailing_slashes() ? trailingslashit($string) : untrailingslashit($string);
    }

    private function wp_template_loader(): void
    {
        global $pagenow;
        $pagenow = 'index.php';
        if (!defined('WP_USE_THEMES')) {
            define('WP_USE_THEMES', true);
        }
        wp();
        require_once ABSPATH . WPINC . '/template-loader.php';
        exit;
    }

    private function new_login_slug($blog_id = '')
    {
        if ($blog_id) {
            $slug = get_blog_option($blog_id, 'cm_hide_login_url');
            if ($slug) {
                return $slug;
            }
        } else {
            $slug = get_option('cm_hide_login_url');
            if ($slug) {
                return $slug;
            }
            if (is_multisite()) {
                if (!function_exists('is_plugin_active_for_network')) {
                    @require_once ABSPATH . 'wp-admin/includes/plugin.php';
                }
                if (defined('CM_HIDE_LOGIN_BASENAME') && function_exists('is_plugin_active_for_network') && is_plugin_active_for_network(CM_HIDE_LOGIN_BASENAME)) {
                    $slug = get_site_option('cm_hide_login_url', 'login');
                    if ($slug) {
                        return $slug;
                    }
                }
            }
        }
        return 'login';
    }

    private function new_redirect_slug(): string
    {
        $slug = get_option('cm_hide_login_redirection_url');
        if ($slug) {
            return $slug;
        }
        if (is_multisite()) {
            if (!function_exists('is_plugin_active_for_network')) {
                @require_once ABSPATH . 'wp-admin/includes/plugin.php';
            }
            if (defined('CM_HIDE_LOGIN_BASENAME') && function_exists('is_plugin_active_for_network') && is_plugin_active_for_network(CM_HIDE_LOGIN_BASENAME)) {
                $slug = get_site_option('cm_hide_login_redirection_url', '404');
                if ($slug) {
                    return $slug;
                }
            }
        }
        return '404';
    }

    public function new_login_url($scheme = null): string
    {
        $url = apply_filters('cm_hide_login_home_url', home_url('/', $scheme));
        if (get_option('permalink_structure')) {
            return $this->with_trailingslash($url . ltrim($this->new_login_slug(), '/'));
        }
        return $url . '?' . $this->new_login_slug();
    }

    public function new_redirect_url($scheme = null): string
    {
        if (get_option('permalink_structure')) {
            return $this->with_trailingslash(home_url('/', $scheme) . ltrim($this->new_redirect_slug(), '/'));
        }
        return home_url('/', $scheme) . '?' . $this->new_redirect_slug();
    }

    private function current_path(array $request): string
    {
        return isset($request['path']) ? $request['path'] : '';
    }

    private function same_host(string $url): bool
    {
        $site = wp_parse_url(home_url());
        $target = wp_parse_url($url);
        if (empty($target['host']) || empty($site['host'])) {
            return true;
        }
        return strcasecmp($target['host'], $site['host']) === 0;
    }

    private function on_new_login_path(array $request): bool
    {
        $path = $this->with_trailingslash($this->current_path($request));
        $login_rel = $this->with_trailingslash(home_url($this->new_login_slug(), 'relative'));
        return $path === $login_rel || (!get_option('permalink_structure') && isset($_GET[$this->new_login_slug()]) && $_GET[$this->new_login_slug()] === '');
    }

    public function redirect_export_data(): void
    {
        if (!empty($_GET) && isset($_GET['action']) && $_GET['action'] === 'confirmation' && isset($_GET['request_id'], $_GET['confirm_key'])) {
            $request_id = (int) $_GET['request_id'];
            $key = sanitize_text_field(wp_unslash($_GET['confirm_key']));
            $result = wp_validate_user_request_key($request_id, $key);
            if (!is_wp_error($result)) {
                $url = add_query_arg(
                    [
                        'action' => 'confirmation',
                        'request_id' => $request_id,
                        'confirm_key' => $key,
                    ],
                    $this->new_login_url()
                );
                wp_safe_redirect($url);
                exit;
            }
        }
    }

    public function getRandomString(int $length = 10): string
    {
        $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $max = strlen($characters) - 1;
        $s = '';
        for ($i = 0; $i < $length; $i++) {
            $s .= $characters[random_int(0, $max)];
        }
        return $s;
    }

    public function on_init(): void
    {
        global $pagenow;

        if (!is_multisite()) {
            $uri = rawurldecode($_SERVER['REQUEST_URI'] ?? '');
            if ((str_contains($uri, 'wp-signup') || str_contains($uri, 'wp-activate')) && apply_filters('cm_hide_login_signup_enable', false) === false) {
                wp_die('This feature is not enabled.');
            }
        }

        $request = parse_url(rawurldecode($_SERVER['REQUEST_URI'] ?? '')) ?: [];
        $path = $this->current_path($request);
        $uri = rawurldecode($_SERVER['REQUEST_URI'] ?? '');

        if ((str_contains($uri, 'wp-login.php') || ($path && untrailingslashit($path) === site_url('wp-login', 'relative'))) && !is_admin()) {
            $this->wp_login_php = true;
            $_SERVER['REQUEST_URI'] = $this->with_trailingslash('/' . str_repeat('-/', 10));
            $pagenow = 'index.php';
            return;
        }

        if (
            ($path && untrailingslashit($path) === home_url($this->new_login_slug(), 'relative')) ||
            (!get_option('permalink_structure') && isset($_GET[$this->new_login_slug()]) && $_GET[$this->new_login_slug()] === '')
        ) {
            $pagenow = 'wp-login.php';
            return;
        }

        if ((str_contains($uri, 'wp-register.php') || ($path && untrailingslashit($path) === site_url('wp-register', 'relative'))) && !is_admin()) {
            $this->wp_login_php = true;
            $_SERVER['REQUEST_URI'] = $this->with_trailingslash('/' . str_repeat('-/', 10));
            $pagenow = 'index.php';
            return;
        }
    }

    public function on_setup_theme(): void
    {
        global $pagenow;
        if (!is_user_logged_in() && $pagenow === 'customize.php') {
            wp_die('This has been disabled', '', 403);
        }
    }

    public function on_wp_loaded(): void
    {
        global $pagenow;

        if (
            (defined('REST_REQUEST') && REST_REQUEST) ||
            (defined('XMLRPC_REQUEST') && XMLRPC_REQUEST) ||
            (defined('DOING_CRON') && DOING_CRON) ||
            (defined('DOING_AJAX') && DOING_AJAX)
        ) {
            return;
        }

        $request = parse_url(rawurldecode($_SERVER['REQUEST_URI'] ?? '')) ?: [];
        $path = $this->current_path($request);

        do_action('cm_hide_login_before_redirect', $request);

        if (!(isset($_GET['action']) && $_GET['action'] === 'postpass' && isset($_POST['post_password']))) {
            if (
                is_admin()
                && !is_user_logged_in()
                && !defined('WP_CLI')
                && $pagenow !== 'admin-post.php'
                && $path !== '/wp-admin/options.php'
                && !$this->on_new_login_path($request)
            ) {
                wp_safe_redirect($this->new_redirect_url());
                exit;
            }

            if (!is_user_logged_in() && isset($_GET['wc-ajax']) && $pagenow === 'profile.php') {
                wp_safe_redirect($this->new_redirect_url());
                exit;
            }

            if (!is_user_logged_in() && $path === '/wp-admin/options.php') {
                wp_safe_redirect($this->new_redirect_url());
                exit;
            }

            if ($pagenow === 'wp-login.php' && $path && $path !== $this->with_trailingslash($path) && get_option('permalink_structure')) {
                $to = $this->with_trailingslash($this->new_login_url());
                if (!empty($_SERVER['QUERY_STRING'])) {
                    $to .= '?' . $_SERVER['QUERY_STRING'];
                }
                wp_safe_redirect($to);
                exit;
            } elseif ($this->wp_login_php) {
                $ref = wp_get_referer();
                if ($ref && str_contains($ref, 'wp-activate.php')) {
                    $ref_parts = parse_url($ref) ?: [];
                    if (!empty($ref_parts['query'])) {
                        parse_str($ref_parts['query'], $args);
                        @require_once WPINC . '/ms-functions.php';
                        if (!empty($args['key'])) {
                            $result = wpmu_activate_signup($args['key']);
                            if (is_wp_error($result)) {
                                $code = $result->get_error_code();
                                if ($code === 'already_active' || $code === 'blog_taken') {
                                    $to = $this->new_login_url();
                                    if (!empty($_SERVER['QUERY_STRING'])) {
                                        $to .= '?' . $_SERVER['QUERY_STRING'];
                                    }
                                    wp_safe_redirect($to);
                                    exit;
                                }
                            }
                        }
                    }
                }
                $this->wp_template_loader();
            } elseif ($pagenow === 'wp-login.php') {
                $redirect_to = admin_url();
                $requested_redirect_to = isset($_REQUEST['redirect_to']) ? $_REQUEST['redirect_to'] : '';

                if (is_user_logged_in() && !isset($_REQUEST['action'])) {
                    $user = wp_get_current_user();
                    $to = apply_filters('cm_logged_in_redirect', $redirect_to, $requested_redirect_to, $user);
                    wp_safe_redirect($to);
                    exit;
                }

                // 🔧 关键补丁：在引入 wp-login.php 前定义期望变量
                if (!isset($user_login)) {
                    $user_login = '';
                }
                // 注意：核心通常使用 $errors (WP_Error)；但你的警告来自 $error（字符串）。
                if (!isset($error)) {
                    $error = '';
                }
                // 可选：有的版本还会读到 $interim_login、$secure_cookie
                if (!isset($interim_login)) {
                    $interim_login = false;
                }
                if (!isset($secure_cookie)) {
                    $secure_cookie = false;
                }

                @require_once ABSPATH . 'wp-login.php';
                exit;
            }
        }
    }

    public function filter_site_url($url, $path, $scheme, $blog_id)
    {
        return $this->filter_wp_login_php($url, $scheme);
    }

    public function filter_network_site_url($url, $path, $scheme)
    {
        return $this->filter_wp_login_php($url, $scheme);
    }

    public function filter_wp_redirect($location, $status)
    {
        if (str_contains($location, 'https://wordpress.com/wp-login.php')) {
            return $location;
        }
        return $this->filter_wp_login_php($location);
    }

    public function filter_wp_login_php($url, $scheme = null)
    {
        if (str_contains($url, 'wp-login.php?action=postpass')) {
            return $url;
        }

        if (!str_contains($url, 'wp-login.php')) {
            return $url;
        }

        $ref = wp_get_referer();
        if ($ref && str_contains($ref, 'wp-login.php')) {
            return $url;
        }

        if (!$this->same_host($url)) {
            return $url;
        }

        if (is_ssl()) {
            $scheme = 'https';
        }

        $parts = explode('?', $url, 2);
        if (isset($parts[1])) {
            parse_str($parts[1], $args);
            if (isset($args['login'])) {
                $args['login'] = rawurlencode($args['login']);
            }
            $url = add_query_arg($args, $this->new_login_url($scheme));
        } else {
            $url = $this->new_login_url($scheme);
        }

        return $url;
    }

    public function filter_welcome_email($value)
    {
        if (is_string($value) && str_contains($value, 'wp-login.php')) {
            $login = wp_parse_url(wp_login_url());
            if (!empty($login['path']) && $login['path'] === '/wp-login.php') {
                $new = '/' . ltrim($this->new_login_slug(), '/');
                $value = str_replace('/wp-login.php', $this->with_trailingslash($new), $value);
            } else {
                $value = str_replace('wp-login.php', $this->with_trailingslash($this->new_login_slug()), $value);
            }
        }
        return $value;
    }

    public function forbidden_slugs(): array
    {
        $wp = new \WP;
        $reserved = array_merge($wp->public_query_vars, $wp->private_query_vars);
        $extra = [
            'wp-admin',
            'wp-login.php',
            'wp-login',
            'wp-json',
            'xmlrpc.php',
            'xmlrpc',
            'feed',
            'page',
            'category',
            'tag',
            'author',
            'search',
            'login',
            'admin'
        ];
        return array_unique(array_merge($reserved, $extra));
    }

    public function add_settings_page()
    {
        add_options_page(
            'CM Hide Login',
            'CM Hide Login',
            'manage_options',
            'cm_settings',
            [$this, 'render_settings_page']
        );
    }

    public function render_settings_page()
    {
        echo 'CM Hide Login';
    }

    public function filter_login_url($login_url, $redirect, $force_reauth)
    {
        if (is_404()) {
            return '#';
        }
        if ($force_reauth === false) {
            return $login_url;
        }
        if (empty($redirect)) {
            return $login_url;
        }
        $r = explode('?', $redirect);
        if ($r[0] === admin_url('options.php')) {
            $login_url = admin_url();
        }
        return $login_url;
    }

    public function filter_user_request_action_email_content($email_text, $email_data)
    {
        if (isset($email_data['confirm_url'])) {
            $login_url = $this->new_login_url();
            $parsed = wp_parse_url($email_data['confirm_url']);
            if ($parsed) {
                $args = [];
                if (!empty($parsed['query'])) {
                    parse_str($parsed['query'], $args);
                }
                $confirm = add_query_arg($args, $login_url);
                $email_text = str_replace('###CONFIRM_URL###', esc_url_raw($confirm), $email_text);
            }
        }
        return $email_text;
    }
}

$hide_login_url = get_option('cm_hide_login_url');
if (!empty($hide_login_url)) {
    $CM_HIDE_LOGIN_URL = new CM_HIDE_LOGIN_URL();
    $CM_HIDE_LOGIN_URL->init();
}